%% AUTHOR: Michal Marencak
% TITLE: COVID-19 epidemic in Slovakia
% through the lens of a parsimonious behavioral SIR model
% National Bank of Slovakia Working paper 02/2022

% !!!
% Not to use without proper citing and referring to the original publication
% !!!

% household keeping
clear;close all;clc


%% Model setup
forecasting_yes_no      = 0; % 1=forecasting; 0=no-forecasting
continue_vacc           = 1; % if 1, after 31.08.2021 vaccination will continue at the past pace
fatigue_again           = 1;

BSIR                    = 1; % transmission reacts to the number of hospitalized cases
seasonality_switcher    = 1; % seasonality on if 1, off otherwise
vaccination_switcher    = 1; % vaccination on if 1, off otherwise
fatigue_refresh         = 1; % removes the fatigue by July 2021

saving_single_figures   = 0;
figure_N                = 0;
final_subplot           = 1;
seasonality_figure      = 1;
fatigue_figure          = 1;
figure_mobility         = 0;
figure_death_rate       = 1;

%% Let's start
load('pandemic_time_series_to_match.mat')

which_period = dates_hosp>='01.07.2020' & (dates_hosp<='28.02.2022');
date_time = dates_hosp(which_period);

t1 = date_time(end);
projection_time = (t1:(t1+calyears))';
projection_time = projection_time(2:end);

%!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
if forecasting_yes_no == 1
    time = [date_time; projection_time];
else
    time = date_time;
end
%!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

period = size(time,1);

%% Parameters
gammaI                  = 0.2;      % recovery rate of infectious not being at hospital
gammaH                  = 0.083;    % recovery rate of infectious at hospital
R0                      = 1.415;
factor_trans_alpha      = 1.6;
factor_trans_delta      = 2.25;
factor_trans_omicron    = 3.72; 
R0_delta                = factor_trans_delta*R0;% >> variant 3times more transmissable 
betabar                 = R0*gammaI; %>>> calibrated such that betabar = R0*gammaI >>> and R0 = 1;
betabar_delta           = R0_delta*gammaI;
beta_vacc_fct           = 3;
days_loss_imunity       = 4000;
lambda                  = 0.021;    % rate of infectious who are hospitalized every period
lambda_vacc_fct         = 3;
lambda_omicron = 0.005;
delta                   = 0.01;  % death rate
delta_vacc_fct          = 13;
delta_omicron           = 0.015;  % death rate
omega_var               = zeros(period,1);
omega                   = 0.0026;
factor_testing          = 0.69; 
factor_testing2         = 1; 


if vaccination_switcher == 1
    if continue_vacc == 1
        omega_var(time>='01.01.2021' & time<'30.07.2021') = omega;
        omega_var(time>='30.07.2021' & time<'22.11.2021') = omega/3;
        omega_var(time>='23.11.2021' & time<'01.01.2022') = omega/2;
        omega_var(time>='01.01.2022') = omega/3;
    else
        omega_var(time>='01.01.2021' & time <'31.12.2021') = omega/3;
    end
end

%% FATIGUE
if BSIR == 1
    fatiguemean = 178;
    fatiguemean_correction = 20;
    fatiguemean_correction2 = 170;
    fatiguemean_again = 150;
    fatiguesig  = 4;
    kappa_bar   = 70;
    kappa_bar2   = kappa_bar*1.165;
    kappa_bar3   = kappa_bar*3;
    fatiguesize = 1;

    
    if fatigue_refresh == 1
        kappa_t = kappa_bar*(1-normcdf((1:period),fatiguemean,fatiguesig))+fatiguesize*kappa_bar*normcdf((1:period),fatiguemean,fatiguesig)...
            +kappa_bar2*normcdf((1:period),fatiguemean+fatiguemean_correction,fatiguesig)-fatiguesize*kappa_bar*normcdf((1:period),fatiguemean+fatiguemean_correction,fatiguesig)...
            -0.1*kappa_bar2*normcdf((1:period),fatiguemean+fatiguemean_correction+50,fatiguesig)...    
        +kappa_bar3*normcdf((1:period),fatiguemean+fatiguemean_correction2,fatiguesig);
    else
        kappa_t = kappa_bar*(1-normcdf((1:period),fatiguemean,fatiguesig))+fatiguesize*kappa_bar*normcdf((1:period),fatiguemean,fatiguesig);
    end
    
    if fatigue_again == 1
        kappa_t(time>='26.10.2021') = 0.3*kappa_t(time>='26.10.2021');
        kappa_t(time>='17.11.2021') = 0.5*kappa_t(time>='17.11.2021');
        kappa_t(time>='30.01.2022') = 2*kappa_t(time>='30.01.2022');
    end
    
    if fatigue_figure == 1
        fig_fatigue = figure;
        plot(time,kappa_t,'b','LineWidth',2)
        hold on
        ylim([0.8*fatiguesize 1.2*max(kappa_t)])
        hold off
        title('Pandemic fatigue')
        xlim([time(1) time(end)])
        grid on
    end
else
    kappa_t = zeros(period,1);
end

%% SEASONALITY
seasonalsize            = -0.1;
seasonalposition        = 0;

if seasonality_switcher == 1
    psi1        = seasonalsize*(cos(((1:period)+seasonalposition)*2*pi/365)-1)/2;
else
    psi1        = 0*seasonalsize*(cos(((1:period)+seasonalposition)*2*pi/365)-1)/2;
end

beta_no_endog = betabar*exp(psi1);

if seasonality_figure == 1
    figure
    subplot(2,2,1)
        plot(time,psi1)
        title('psi')
        grid on
    subplot(2,2,2)
        plot(time,exp(psi1))
        title('exp(psi)')
        grid on
    subplot(2,2,3)
        plot(time,betabar*exp(psi1)/gammaI)
        hold on
        plot(time,betabar_delta*exp(psi1)/gammaI)
        hold off
        title('Basic reproduction number without endogenous decline in \beta')
        grid on
    subplot(2,2,4)
        plot(time,beta_no_endog)
        title('\beta without endogenous response')
        grid on
end

%% initial calibration of infectious people
I0 = sk_cases_final(1,1);

% Variables
N           = [1; zeros(period-1,1)];
Iu          = [I0; zeros(period-1,1)];
Iv          = zeros(period,1);
I           = Iu+Iv;
Iu_alpha    = zeros(period,1);
Iv_alpha    = Iu_alpha;
I_alpha     = Iu_alpha;
Iu_delta    = zeros(period,1);
Iv_delta    = Iu_delta;
I_delta     = Iu_delta;
Iu_omicron  = zeros(period,1);
Iv_omicron  = Iu_delta;
I_omicron   = Iu_delta;
S           = N-Iu;
R           = [zeros(period,1)];
Rdaily      = R;
D           = [zeros(period,1)];
Ku          = [zeros(period,1)];
Ku_alpha    = Ku;
Ku_delta    = Ku;
Ku_omicron  = Ku;
Kv          = Ku;
Kv_alpha    = Ku;
Kv_delta    = Ku;
Kv_omicron  = Ku;
K_total     = Ku;
V           = R;
Vdaily      = V;
V_hypothetical_first_doses = V;
V_3rd = V;
V_non3rd = V;
Hu          = R;
Hu_other    = R;
Hu_omicron  = R;
Hv          = R;
Hv_other    = R;
Hv_omicron  = R;
H           = R;
A           = H;
Disch       = A;
beta        = [betabar*exp(psi1(1)); zeros(period-1,1)];
beta_alpha  = zeros(period,1);
beta_delta  = zeros(period,1);
beta_omicron  = zeros(period,1);
beta_vacc = zeros(period,1);

%beta = ones(period,1)*0.8;

a = 1;

delta_used = [delta; zeros(period-1,1)];

% Solving recursively the system of recurrence equations
for t=1:period-1
    
    if (time(t+1) == '14.11.2020')
        Iu_alpha(t)     = 1/5451733;
        I_alpha(t)      = Iu_alpha(t)+Iv_alpha(t);
        S(t)            = S(t)-1/5451733;
    end
    
    if (time(t+1) == '23.05.2021')
        Iu_delta(t)   = 1/5451733;
        I_delta(t)    = Iu_delta(t)+Iv_delta(t);
        S(t)          = S(t)-1/5451733;
    end

    if (time(t+1) == '10.11.2021')
        Iu_omicron(t)   = 1/5451733;
        I_omicron(t)    = Iu_omicron(t)+Iv_omicron(t);
        S(t)          = S(t)-1/5451733;
    end
    
    delta_used(t+1) = delta;
    
    % beta = reduced-form for the behavioral response of the transmission rate to the level of hospitalizations

    if (time(t+1) < '24.10.2020')
        beta(t+1) = betabar*exp(psi1(t));
    elseif (time(t+1) >= '24.10.2020') && (time(t+1) < '19.11.2020')
        delta   = 0.012;
        gammaH = 0.081;
        delta_vacc = delta/delta_vacc_fct;
        beta(t+1) = factor_testing*betabar*exp(psi1(t));
    elseif (time(t+1) >= '19.11.2020') && (time(t+1) < '08.12.2020')
        delta = 0.012;
        gammaH = 0.081;
        beta(t+1) = (1-a)*beta(t)+a*betabar*exp(-kappa_t(t)*sum(H(max(1,t-20):max(1,t-10))) +psi1(t));    
    elseif (time(t+1) >= '08.12.2020') && (time(t+1) < '15.01.2021')
        delta = 0.024;
        gammaH = 0.069;
        beta(t+1) = (1-a)*beta(t)+a*betabar*exp(-kappa_t(t)*sum(H(max(1,t-20):max(1,t-10))) +psi1(t));  
    elseif (time(t+1) >= '15.01.2021') && (time(t+1) < '28.01.2021')
        delta = 0.023;
        gammaH = 0.070;
        beta(t+1) = (1-a)*beta(t)+a*betabar*exp(-kappa_t(t)*sum(H(max(1,t-20):max(1,t-10))) +psi1(t));  
    elseif (time(t+1) >= '28.01.2021') && (time(t+1) < '12.02.2021')
          beta(t+1) = (1-a)*beta(t)+factor_testing2*a*betabar*exp(-kappa_t(t)*sum(H(max(1,t-20):max(1,t-10))) +psi1(t));  
    elseif (time(t+1) >= '12.02.2021') && (time(t+1) < '30.06.2021')
          beta(t+1) = (1-a)*beta(t)+factor_testing2*a*betabar*exp(-kappa_t(t)*sum(H(max(1,t-20):max(1,t-10))) +psi1(t));  
    elseif (time(t+1) >= '30.06.2021') && (time(t+1) < '05.02.2022')
          beta(t+1) = (1-a)*beta(t)+factor_testing2*a*betabar*exp(-kappa_t(t)*sum(H(max(1,t-20):max(1,t-10))) +psi1(t));  
          lambda  = 0.015;
          gammaH = 0.099;
          gammaI = 0.3;
          delta = 0.026;
          beta(t+1) = (1-a)*beta(t)+a*betabar*exp(-kappa_t(t)*sum(H(max(1,t-20):max(1,t-10))) +psi1(t));
    else
          lambda  = 0.015;
          gammaH = 0.099;
          gammaI = 0.3;
          delta = 0.026;
          beta(t+1) = (1-a)*beta(t)+a*betabar*exp(-kappa_t(t)*sum(H(max(1,t-20):max(1,t-10))) +psi1(t));
    end
    
    % regular testing
    if (time(t+1) >= '01.04.2021') && (time(t+1) < '20.06.2021')
        beta(t+1)=0.163;
    end
    
    % share of boosters among those losing their vaccination immunity
    omega_3rd = 1;

    beta_alpha(t+1) = factor_trans_alpha*beta(t+1);
    beta_delta(t+1) = factor_trans_delta*beta(t+1);
    beta_omicron(t+1) = factor_trans_omicron*beta(t+1);
    if (time(t+1) < '30.07.2021')
        beta_vacc(t+1)  = beta(t+1)/beta_vacc_fct;
    else
        beta_vacc(t+1)  = beta_delta(t+1)/beta_vacc_fct;
    end
    % -----
    Ku(t)           = beta(t) * S(t) * I(t);
    Kv(t)           = beta_vacc(t) * V(t) * I(t);
    Ku_alpha(t)     = beta_alpha(t) * S(t) * I_alpha(t);
    Kv_alpha(t)     = beta_vacc(t) * V(t) * I_alpha(t);
    Ku_delta(t)     = beta_delta(t) * S(t) * I_delta(t);
    Kv_delta(t)     = beta_vacc(t) * V(t) * I_delta(t);
    Ku_omicron(t)     = beta_omicron(t) * S(t) * I_omicron(t);
    Kv_omicron(t)     = beta_vacc(t) * V(t) * I_omicron(t);
    K_total(t) = Ku(t) + Kv(t) + Ku_alpha(t) + Kv_alpha(t) + Ku_delta(t) + Kv_delta(t) + Ku_omicron(t) + Kv_omicron(t);
    % -----
    S(t+1)          = S(t) - ( Ku(t)+Ku_alpha(t)+Ku_delta(t)+Ku_omicron(t) ) - omega_var(t)*(S(t)/(S(t)+V_non3rd(t)))*S(t)+Rdaily(max(1,t-days_loss_imunity))... 
         +Vdaily(max(1,t-days_loss_imunity))*(1-omega_3rd); 
    % -----
    Iu(t+1)         = Iu(t) + Ku(t) - (lambda + gammaI)* Iu(t);
    Iv(t+1)         = Iv(t) + Kv(t) - (lambda/lambda_vacc_fct + gammaI)* Iv(t);
    Iu_alpha(t+1)   = Iu_alpha(t) + Ku_alpha(t) - (lambda + gammaI)* Iu_alpha(t);
    Iv_alpha(t+1)   = Iv_alpha(t) + Kv_alpha(t) - (lambda/lambda_vacc_fct + gammaI)* Iv_alpha(t);
    Iu_delta(t+1)   = Iu_delta(t) + Ku_delta(t) - (lambda + gammaI)* Iu_delta(t);
    Iv_delta(t+1)   = Iv_delta(t) + Kv_delta(t) - (lambda/lambda_vacc_fct + gammaI)* Iv_delta(t);
    Iu_omicron(t+1)   = Iu_omicron(t) + Ku_omicron(t) - (lambda_omicron + gammaI)* Iu_omicron(t);
    Iv_omicron(t+1)   = Iv_omicron(t) + Kv_omicron(t) - (lambda_omicron/lambda_vacc_fct + gammaI)* Iv_omicron(t);
    I(t+1)          = Iu(t+1)+ Iv(t+1);
    I_alpha(t+1)    = Iu_alpha(t+1)+ Iv_alpha(t+1);
    I_delta(t+1)    = Iu_delta(t+1)+Iv_delta(t+1);
    I_omicron(t+1)    = Iu_omicron(t+1)+Iv_omicron(t+1);
    % -----
    Hu_other(t+1)   = Hu_other(t) + lambda*(Iu(t)+Iu_alpha(t)+Iu_delta(t)) - (delta + gammaH)* Hu_other(t);
    Hu_omicron(t+1) = Hu_omicron(t) + lambda_omicron*Iu_omicron(t) - (delta_omicron + gammaH)* Hu_omicron(t);
    Hu(t+1)         = Hu_omicron(t+1) + Hu_other(t+1);
    Hv_other(t+1)   = Hv_other(t) + lambda/lambda_vacc_fct*(Iv(t)+Iv_alpha(t)+Iv_delta(t)) - (delta/delta_vacc_fct + gammaH)* Hv_other(t);
    Hv_omicron(t+1) = Hv_omicron(t) + lambda_omicron/lambda_vacc_fct*Iv_omicron(t) - (delta_omicron/delta_vacc_fct + gammaH)* Hv_omicron(t);
    Hv(t+1)         = Hv_other(t+1) + Hv_omicron(t+1);
    H(t+1)          = Hu(t+1) + Hv(t+1);
    A(t+1) = lambda*(Iu(t)+Iu_alpha(t)+Iu_delta(t))+ lambda_omicron*Iu_omicron(t)+lambda/lambda_vacc_fct*(Iv(t)+Iv_alpha(t)+Iv_delta(t))+ lambda_omicron/lambda_vacc_fct*Iv_omicron(t);
    Disch(t+1) = (delta + gammaH)* Hu_other(t)+(delta_omicron + gammaH)* Hu_omicron(t)+(delta/delta_vacc_fct + gammaH)* Hv_other(t)+(delta_omicron/delta_vacc_fct + gammaH)* Hv_omicron(t);
    % -----
    R(t+1)          = R(t) + gammaI*(Iu(t)+Iu_alpha(t)+Iu_delta(t)+Iu_omicron(t)+Iv(t)+Iv_alpha(t)+Iv_delta(t)+Iv_omicron(t)) +...
        gammaH*Hu(t) + gammaH*Hv(t) - omega_var(t)*R(t)-Rdaily(max(1,t-days_loss_imunity));
    Rdaily(t+1)     = R(t+1)-R(t);
    % -----
    V_hypothetical_first_doses(t+1) = V_hypothetical_first_doses(t) + omega_var(t)*S(t)+omega_var(t)*R(t);
    V(t+1)          = V(t) + omega_var(t)*(S(t)/(S(t)+V_non3rd(t)))*S(t)+omega_var(t)*R(t)-Kv(t)-Kv_alpha(t)-Kv_delta(t)-Kv_omicron(t)-Vdaily(max(1,t-days_loss_imunity))*(1-omega_3rd);
    Vdaily(t+1)     = V(t+1)-V(t);
    V_3rd(t+1)      = V_3rd(t)+omega_3rd*Vdaily(max(1,t-days_loss_imunity));
    V_non3rd(t+1)   = V_non3rd(t)+(1-omega_3rd)*Vdaily(max(1,t-days_loss_imunity));
    % -----
    D(t+1)          = D(t) + delta*Hu_other(t) + delta/delta_vacc_fct*Hv_other(t) + delta_omicron*Hu_omicron(t) + delta_omicron/delta_vacc_fct*Hv_omicron(t);
    % -----
    N(t+1)          = S(t+1)  + I(t+1) + I_alpha(t+1) + I_delta(t+1) + I_omicron(t+1) + H(t+1) + R(t+1) + V(t+1) + D(t+1);
end


%% Results
x1 = projection_time(1); x2 = projection_time(end);
y1 = 0; y2 = 5;

% Reporting
rmse_hosp = sqrt( sum((Hu(1:size(sk_hosp(which_period),1))-sk_hosp(which_period)).^2) );
rmse_cum_deaths = sqrt( sum((D(1:size(sk_hosp(which_period),1))-cumsum(sk_deaths_final(which_period))).^2) );
disp(['RMSE of hospitalizations = ' num2str(rmse_hosp)])
disp(['RMSE of cumulated deaths = ' num2str(rmse_cum_deaths)])
disp(['factor testing = ' num2str(factor_testing)])

%% Final figure hospitalizations simple
final_figure_hosp = figure;
sk_ag_cases_final(isnan(sk_ag_cases_final)) = 0;
sk_cases_total = sk_cases_final(which_period);%+sk_ag_cases_final(which_period);
    hold on
    plot(date_time,sk_hosp(which_period)*sk_inh,'--r','LineWidth',2)
    plot(time,H*sk_inh,'b','LineWidth',2)
    hold off
    grid on
    xlim([time(1) time(end)])
    ylim([0 1.2*max(max(H*sk_inh),max(sk_hosp(which_period)*sk_inh))])
    legend('data', 'model')
    ylabel('Number pf persons')
    title('Hospitalizations')

%% final figure cumulative deaths
final_figure_cumdeaths = figure;
    hold on
    plot(time,sk_inh*D,'b','LineWidth',2)
    plot(date_time,cumsum(sk_inh*sk_deaths_final(which_period)),'--r','LineWidth',2)
    hold off
    grid on
    xlim([time(1) time(end)])
    ylim([0 1.2*max(sk_inh*max(D),max(sk_inh*cumsum(sk_deaths_final(which_period))))])     
    ylabel('Number pf persons')
    title('Cumulative deaths')

%% final figure vaccination
 final_figure_vacc = figure;
    hold on
    plot(date_time,sk_vacc_final(which_period),'--r', 'LineWidth', 2)
    plot(time,V_hypothetical_first_doses,'b','LineWidth',2)
    hold off
    grid on
    xlim([time(1) time(end)])
    ylim([0 1.2*max(max(V_hypothetical_first_doses),max(sk_vacc_final(which_period)))])
    ylabel('Share of population') 
    legend('data', 'model','location','northwest') 
    title('Share of vaccinated')

%% final figure variant dominance
final_figure_var_dominance = figure;
    hold on
    plot(time(1:end-1),(Ku(1:end-1)+Kv(1:end-1))./K_total(1:end-1),':k','LineWidth',2)
    plot(time(1:end-1),(Ku_alpha(1:end-1)+Kv_alpha(1:end-1))./K_total(1:end-1),'-b','LineWidth',2)
    plot(time(1:end-1),(Ku_delta(1:end-1)+Kv_delta(1:end-1))./K_total(1:end-1),'--k','LineWidth',2)
    plot(time(1:end-1),(Ku_omicron(1:end-1)+Kv_omicron(1:end-1))./K_total(1:end-1),'-.m','LineWidth',2)
    hold off
    title('Variant dominance')
    grid on
    xlim([time(1) time(end)])
    ylim([0 1.2*max(beta/gammaI)]) 
    legend('initial strain','alpha','delta','omicron','location','SouthWest')  

%% share of vaccinated among hospitalizations
final_figure_hosp_vacc = figure;
    hold on
    plot(time,Hv./H*100,'k','LineWidth',2)
    hold off
    grid on
    xlim([time(1) time(end)])
    title('Share of vaccinated among hospitalized')
    ylabel('Percentage points')    
        
%% final figure hospitalizations of vaccinated
hosp_from_admission = cumsum(sk_admission_final)-cumsum(sk_discharges_final);
final_figure_hosp_vacc_unvacc_total = figure;
    hold on
    plot(date_time,sk_hosp(which_period)*sk_inh,'--r','LineWidth',2)
    plot(date_time,movmean(hosp_from_admission(which_period)*sk_inh,7),':k','LineWidth',2)
    hold off
    grid on
    xlim([time(1) time(end)])
    ylim([0 1.2*max(max(Hu*sk_inh),max(sk_inh*sk_hosp(which_period)))]) 
    title('Hospitalized')
    ylabel('Number pf persons')
    legend('data NCZI','data admissions-discharges')

%% final figure hospitalizations of vaccinated
hosp_from_admission = cumsum(sk_admission_final)-cumsum(sk_discharges_final);
final_figure_hosp_vacc_unvacc_total = figure;
    hold on
    plot(date_time,sk_hosp(which_period)*sk_inh,'--r','LineWidth',2)
    plot(date_time,movmean(hosp_from_admission(which_period)*sk_inh,7),':k','LineWidth',2)
    plot(time,Hu*sk_inh,'k','LineWidth',2)
    plot(time,Hv*sk_inh,'m','LineWidth',2)
    plot(time,H*sk_inh,'b','LineWidth',2)
    a = fill([x1 x1 x2 x2],[y1 y2 y2 y1],'y','LineStyle','none','FaceColor',[0.5 0.5 0.5]);
    a.FaceAlpha = 0.2; 
    hold off
    grid on
    xlim([time(1) time(end)])
    ylim([0 1.2*max(max(Hu*sk_inh),max(sk_inh*sk_hosp(which_period)))]) 
    title('Hospitalized')
    ylabel('Number pf persons')
    legend('data NCZI','data admissions-discharges','unvaccinated','vaccinated','total')    
        
%% final figure admissions  
final_figure_admissions = figure;
    hold on
    plot(date_time,movmean(sk_admission_final(which_period),7)*sk_inh,'--r','LineWidth',2)
    plot(time,A*sk_inh,'b','LineWidth',2)
    hold off
    grid on
    xlim([time(1) time(end)])
    legend('data','model','location','NorthWest')
    ylabel('Number of persons')    
    title('Admissions')  
    
%% final figure discharges
final_figure_discharges = figure;
    hold on
    plot(date_time,movmean(sk_discharges_final(which_period),7)*sk_inh,'--r','LineWidth',2)
    plot(time,Disch*sk_inh,'b','LineWidth',2)
    hold off
    grid on
    xlim([time(1) time(end)])
    ylabel('Number of persons')    
    title('Discharges')

%% final figure cases 
final_figure_daily_cases = figure;
    hold on
    plot(time(1:end-1),Ku(1:end-1),'b','LineWidth',2)
    plot(time(1:end-1),Kv(1:end-1),'-.b','LineWidth',2)
    plot(time(1:end-1),Ku_alpha(1:end-1),'k','LineWidth',2)
    plot(time(1:end-1),Kv_alpha(1:end-1),'-.k','LineWidth',2)
    plot(time(1:end-1),Ku_delta(1:end-1),'r','LineWidth',2)
    plot(time(1:end-1),Kv_delta(1:end-1),'-.r','LineWidth',2)
    plot(time(1:end-1),Ku_omicron(1:end-1),'c','LineWidth',2)
    plot(time(1:end-1),Kv_omicron(1:end-1),'-.c','LineWidth',2)
    hold off
    legend('initial variant unvaccinated','initial variant vaccinated',...
        'alpha unvaccinated','alpha vaccinated',...
        'delta variant unvaccinated','delta vaccinated',...
        'omicron variant unvaccinated','omicron vaccinated',...
        'location','NorthWest')
    grid on
    xlim([time(1) time(end)])
    ylim([0 1.2*max(max(sk_cases_total),max(Ku_omicron(1:end-1)))])
    title('Number of daily new cases - model dynamics') 

%% death rate figure
final_figure_daily_deaths = figure;
    hold on
    plot(date_time,sk_deaths_final(which_period),'--r','LineWidth',2)    
    plot(time,[0;diff(D)],'b','LineWidth',2)
    hold off
    grid on
    xlim([time(1) time(end)])
    ylim([0 1.2*max(diff(D))])     
    title('Daily deaths')
